//Comment out this define to disable normals.
//Note that normals must be enabled for real time lighting.
//
#define NORMALS
//
//This is done in a #define because there is a large number of points where
//a run-time decision branch might occur, and code is often grouped on a by-operation
//basis. Generally, everything having to do with normals is done immediately after
//anything to do with vertices so that the code that processed the vertex data may
//then process normal data without CPU cache swapping.

using UnityEngine;
using System.Collections.Generic;

/*
  Toby Grierson's turboslice kit
  
  The public methods are:

    GameObject[] splitByLine(GameObject target, Camera camera, Vector3 _start, Vector3 _end)
    GameObject[] splitByTriangle(GameObject target, Vector3[] triangleInWorldSpace)
    GameObject[] splitByPlane(GameObject target, Vector4 localPlane)

  We give it a target object (which includes a single mesh renderer with a single material), the camera and
  the start and end positions of the touch on-screen.
  
  Or, in the second, a plane in world space defined by three vertices.
  
  Finally, in the third, a vector4 (equation of plane) describing the plane in LOCAL space (this is hardcore
  mode, so this math task can be up to you).

  You may (in the Unity editor) add mesh-bearing assets that will be cached on-load for faster processing.
  
  There is a public property; bool meshCaching. Do NOT turn this on unless the diced gameobjects are
  doomed to deletion like in a slicing game. There IS a safety switch that will warn you if it detects inappropriate
  use and turns off the feature, releasing the memory.

  It is not good at quickly identifying when objects will not be affected by the split. This is the responsibility
  of the client; see the demonstration code for a simple example of doing it with colliders.
*/

public class _TurboSliceModuleHack : MonoBehaviour
{
	public const bool supportsSkinned = false;
	
	public GameObject[] splitByPlaneRD(GameObject go, Vector4 plane, bool destroyOriginal)
	{
		return new GameObject[0];
	}
}

public partial class TurboSlice : _TurboSliceModuleHack
{
	private static TurboSlice _instance;
	public static TurboSlice instance {
		get {
			if(_instance == null)
			{
				Debug.LogWarning("No TurboSlice component in scene. Creating one now with default configuration. Add one manually if you wish to configure infill support. Please refer to the guide for more details.");
				GameObject go = new GameObject();
				go.AddComponent<TurboSlice>();
				_instance = go.GetComponent<TurboSlice>();
			}
			return _instance;
		}
	}
	
	public bool meshCaching = false;
	
	//Any object put in here will have its mesh data fetched on boot rather than on first usage in play-time.
	public GameObject[] preload;
	
	[System.Serializable]
	public class InfillConfiguration {
		public Material material;
		public Rect regionForInfill;
	}
	
	public InfillConfiguration[] infills = new InfillConfiguration[0];

	//A const for internal use
	const float epsilon = 0f;

	// Suppose we have source model A and we slice it. We create a mesh cache X (cloned from it) which is going to contain
	//the geometrical for the resulting slices and ALL subsequent slices. A number too large will result in unneccessarily
	//large memory allocations while a figure too small will result in frequent time-consuming reallocations as the vector
	//must be expanded to accomodate the growing geometry. Note that when a lineage is finally, entirely off screen, this
	//memory will be dereferenced such that the VM can release it at its discretion.
	// This number (9/2) was arrived at experimentally; at this number, reallocations are rare ( < 3 times a minute ).
	const float factorOfSafetyGeometry = 9f / 2f;
	
	// This is a little different as indices are -not- retained. This is how much we need to allocate for each resultant mesh,
	//compared to the original. I have set it assume that resultant meshes may be up to 90% the complexity of originals because
	//a highly uneven slice (a common occurrence) will result in this.
	const float factorOfSafetyIndices = 0.9f;
	
	//This here is a mesh cache. A mesh cache contains data for both a collection of slice results (multiples might refer to
	//a single cache) or for a preloaded mesh. When preloaded mesh A is split, it will yield meshs B and C and its mesh
	//cache X will be duplicated into mesh cache Y that meshes B, C and all further derivatives will refer to. When B, C and
	//all other derivatives have fallen away from the screen, their respective mesh cache will be zapped.
	class MeshCache
	{
		public bool wasPreloaded = false;
		
		private float _creationTime = Time.time;
		public float creationTime {
			get {
				return wasPreloaded ? float.MaxValue : _creationTime;
			}
		}
		
		public TurboList<Vector3> vertices;
#if NORMALS
		public TurboList<Vector3> normals;
#endif	
		public TurboList<Vector2> UVs;
		
		public int[][] indices;
	
		public Material[] mats;
		
		public MeshCache clone()
		{
			MeshCache mc = new MeshCache();
			
			int cap = Mathf.RoundToInt((float) vertices.Count * factorOfSafetyGeometry);
			
			mc.vertices = new TurboList<Vector3>(cap);
#if NORMALS
			mc.normals = new TurboList<Vector3>(cap);
#endif
			mc.UVs = new TurboList<Vector2>(cap);
			mc.vertices.AddArray(vertices.array);
#if NORMALS
			mc.normals.AddArray(normals.array);
#endif
			mc.UVs.AddArray(UVs.array);
			
			mc.indices = new int[indices.Length][];
			for(int i = 0 ; i < indices.Length; i++)
			{
				mc.indices[i] = new int[indices[i].Length];
				System.Array.Copy(indices[i], mc.indices[i], indices[i].Length);
			}
			
			mc.mats = mats;
			
			return mc;
		}
	}
	
	private Dictionary<Mesh,MeshCache> meshCaches;
	
	public void releaseCacheByMesh(Mesh m)
	{
		if(meshCaches != null && meshCaches.ContainsKey(m))
		{
			meshCaches.Remove(m);
		}
	}
	
	#if NORMALS
	private void perfectSubset(TurboList<int> _sourceIndices, TurboList<Vector3> _sourceVertices, TurboList<Vector3> _sourceNormals, TurboList<Vector2> _sourceUVs,
		out int[] targetIndices, TurboList<Vector3> targetVertices, TurboList<Vector3> targetNormals, TurboList<Vector2> targetUVs, ref int[] transferTable)
	#else
	private void perfectSubset(TurboList<int> _sourceIndices, TurboList<Vector3> _sourceVertices, TurboList<Vector2> _sourceUVs,
		out int[] targetIndices, TurboList<Vector3> targetVertices, TurboList<Vector2> targetUVs, ref int[] transferTable)
	#endif
	{
		int[] sourceIndices = _sourceIndices.array;
		Vector3[] sourceVertices = _sourceVertices.array;
		Vector2[] sourceUVs = _sourceUVs.array;
		#if NORMALS
		Vector3[] sourceNormals = _sourceNormals.array;
		#endif
		
		targetIndices = new int[_sourceIndices.Count];
		
		int targetIndex = targetVertices.Count;
		for(int i = 0; i < _sourceIndices.Count; i++)
		{
			int requestedVertex = sourceIndices[i];
			
			int j = transferTable[requestedVertex];
			
			if(j == -1)
			{
				j = targetIndex;
				transferTable[requestedVertex] = j;
				targetIndex++;
			}
			
			targetIndices[i] = j;
		}
		
		targetVertices.EnsureCapacity(targetIndex);
		#if NORMALS
		targetNormals.EnsureCapacity(targetIndex);
		#endif
		targetUVs.EnsureCapacity(targetIndex);
		
		targetVertices.Count = targetIndex;
		#if NORMALS
		targetNormals.Count = targetIndex;
		#endif
		targetUVs.Count = targetIndex;
		
		for(int i = 0; i < transferTable.Length; i++)
		{
			int j = transferTable[i];
			if(j != -1)
				targetVertices.array[j] = sourceVertices[i];
		}
		
		#if NORMALS
		for(int i = 0; i < transferTable.Length; i++)
		{
			int j = transferTable[i];
			if(j != -1)
				targetNormals.array[j] = sourceNormals[i];
		}
		#endif
		
		for(int i = 0; i < transferTable.Length; i++)
		{
			int j = transferTable[i];
			if(j != -1)
				targetUVs.array[j] = sourceUVs[i];
		}
	}
	
	// Use this for initialization
	void Start ()
	{
		if(_instance != null)
		{
			Debug.LogWarning("There may be multiple TurboSlice components in scene " + Application.loadedLevelName + ". Please review this!");
		}
		
		_instance = this;
		
		if(meshCaching)
		{
			meshCaches = new Dictionary<Mesh, MeshCache>();
			
			foreach(GameObject victim in preload)
			{
				MeshCache c = cacheFromGameObject(victim, false);
				
				c.wasPreloaded = true;
				
				Mesh m = victim.GetComponent<MeshFilter>().sharedMesh;
				
				meshCaches[m] = c;
			}
		}
	}
	
	private MeshCache cacheFromGameObject(GameObject victim, bool includeRoomForGrowth)
	{
		MeshFilter filter = victim.GetComponent<MeshFilter>();
		
		Mesh m = filter.sharedMesh;
		
		int initialCapacity = includeRoomForGrowth ? Mathf.RoundToInt((float) m.vertexCount * factorOfSafetyGeometry) : m.vertexCount;
		
		MeshCache c = new MeshCache();
		
		c.vertices = new TurboList<Vector3>(initialCapacity);
#if NORMALS
		c.normals = new TurboList<Vector3>(initialCapacity);
#endif
		c.UVs = new TurboList<Vector2>(initialCapacity);
		
		c.indices = new int[m.subMeshCount][];
		
		for(int i = 0; i < m.subMeshCount; i++)
		{
			c.indices[i] = m.GetTriangles(i);
		}
		
		c.vertices.AddArray(m.vertices);	
#if NORMALS
		c.normals.AddArray(m.normals);
#endif
		c.UVs.AddArray(m.uv);
		
		Renderer renderer = victim.GetComponent<MeshRenderer>();
		
		if(renderer != null)
		{
			if(renderer.sharedMaterials == null)
			{
				c.mats = new Material[1];
				c.mats[0] = renderer.sharedMaterial;
			}
			else
			{
				c.mats = renderer.sharedMaterials;
			}
		}
		else
		{
			Debug.LogError("Object '" + victim.name + "' has no renderer");
		}
		
		return c;
	}
	
	private Vector4 planeForTransform(Camera camera, GameObject target, Vector3 _start, Vector3 _end)
	{		
		Ray r;
		
		r = camera.ScreenPointToRay(_start);
		Vector3 start = r.GetPoint(10f);
		
		r = camera.ScreenPointToRay(_end);
		Vector3 end = r.GetPoint(10f);
		
		Vector3 _middle = (_start + _end) / 2f;
		r = camera.ScreenPointToRay(_middle);
		Vector3 middle = r.GetPoint(15f);
		
		List<Vector3> _plane = new List<Vector3>();
		
		_plane.Add(start);
		_plane.Add(middle);
		_plane.Add(end);
			
		Vector4 plane = TurboSlice.planeFromTriangle(target, _plane.ToArray());
		return plane;
	}
	
	//This code here is translated from John Ratcliff's; it converts a triplet of vertices into a four-vector describing a plane.
	//This is used by splitByLine and splitByTriangle to feed a four-vector to splitByPlane.
	private static Vector4 planeFromTriangle(GameObject target, Vector3[] t)
	{	
		Matrix4x4 matrix = target.transform.worldToLocalMatrix;
		
		t[0] = matrix.MultiplyPoint(t[0]);
		t[1] = matrix.MultiplyPoint(t[1]);
		t[2] = matrix.MultiplyPoint(t[2]);
		
		Vector4 p = Vector4.zero;
		
		p.x = t[0].y * (t[1].z - t[2].z) + t[1].y * (t[2].z - t[0].z) + t[2].y * (t[0].z - t[1].z);
		p.y = t[0].z * (t[1].x - t[2].x) + t[1].z * (t[2].x - t[0].x) + t[2].z * (t[0].x - t[1].x);
		p.z = t[0].x * (t[1].y - t[2].y) + t[1].x * (t[2].y - t[0].y) + t[2].x * (t[0].y - t[1].y);
		p.w = -( t[0].x * (t[1].y * t[2].z - t[2].y * t[1].z) + t[1].x * (t[2].y * t[0].z - t[0].y * t[2].z) + t[2].x * (t[0].y * t[1].z - t[1].y * t[0].z) );
		
		return p;
	}
	
	public enum PlaneTriResult {
		PTR_FRONT, PTR_BACK, PTR_SPLIT
	};
	
	private static float classifyPoint(ref Vector4 plane, ref Vector3 p)
	{
    	return p.x * plane.x + p.y * plane.y + p.z * plane.z + plane.w;
	}
	
	private static PlaneTriResult getSidePlane(ref Vector3 p, ref Vector4 plane)
	{
	  double d = distanceToPoint(ref p, ref plane);
	
	  if ( (d+epsilon) > 0 )
			return PlaneTriResult.PTR_FRONT; // it is 'in front' within the provided epsilon value.

	  return PlaneTriResult.PTR_BACK;
	}
	
	private static float distanceToPoint(ref Vector3 p, ref Vector4 plane)
	{
		float d = p.x * plane.x + p.y * plane.y + p.z * plane.z + plane.w;
		return d;
	}
	
	static float intersectCommon(ref Vector3 p1, ref Vector3 p2, ref Vector4 plane)
	{
		float dp1 = distanceToPoint(ref p1, ref plane);
		
		Vector3 dir = p2 - p1;
		
		float dot1 = dir.x * plane.x + dir.y * plane.y + dir.z * plane.z;
		float dot2 = dp1 - plane.w;
		
		float t = -(plane.w + dot2 ) / dot1;
		
		return t;
	}

	/* I'm going to reorganize splitTriangle - which is called per triangle and operates per triangle - into
	 * splitTriangles. splitTriangles operates on a batch and is organized on a per action basis.
	 * 
	 * The first observation about splitTriangle is that classifyPoint is called upon pretty much every single
	 * one going in. Second, some of this work is repeated when indices repeat vertex references...
	 *
	 * Let's assume that it's probably more costly to hunt down repeated vertices than to repeat the work a little.
	 * 
	 */

	struct SplitAction
	{
		public const short nullIndex = -1;
		
		public const byte TO_FRONT = 0x01, TO_BACK = 0x02, INTERSECT = 0x04;
		
		public byte flags;			//1
		public int cloneOf;
		public int index0, index1, realIndex, parentTriangleIndex;	//4 * 4 = 16 bytes
		public float intersectionResult;							//4 bytes
		
		public SplitAction(bool _toFront, bool _toBack, int _index0)
		{
			flags = 0;
			if(_toFront) flags = (byte) (flags | TO_FRONT);
			if(_toBack) flags = (byte) (flags | TO_BACK);
			parentTriangleIndex = -1;
			index0 = _index0;
			index1 = nullIndex;
			cloneOf = nullIndex;
			realIndex = index0;
			intersectionResult = 0f;
		}
		
		public SplitAction(int _index0, int _index1, int _parentTriangleIndex)
		{
			flags = TO_FRONT | TO_BACK | INTERSECT;
			parentTriangleIndex = _parentTriangleIndex;
			index0 = _index0;
			index1 = _index1;
			cloneOf = nullIndex;
			realIndex = nullIndex;
			intersectionResult = 0f;
		}
		
		public new string ToString()
		{
			return string.Format("SplitAction: Parent={0}, Geometry={1}", parentTriangleIndex, realIndex);
		}
	};
	
	static void splitTriangles(Vector4 plane, int[] sourceIndices, MeshCache meshCache, InfillConfiguration infill, TurboList<int> frontIndices, TurboList<int> backIndices)
	{
		bool doInfill = infill != null;
		
		Vector3[] sourceGeometry = meshCache.vertices.array;
		#if NORMALS
		Vector3[] sourceNormals = meshCache.normals.array;
		#endif
		Vector2[] sourceUVs = meshCache.UVs.array;
		
		float[] pointClassifications = new float[sourceIndices.Length];
		for(int i = 0; i < pointClassifications.Length; i++)
		{
			pointClassifications[i] = classifyPoint(ref plane, ref sourceGeometry[ sourceIndices[i] ]);
		}
		
		//Now we're going to do the decision making pass. This is where we assess the side figures and produce actions...
		
		int inputTriangleCount = sourceIndices.Length / 3;
		
		//A good action count estimate can avoid reallocations.
		//We expect exactly five actions per triangle.
		int actionEstimate = inputTriangleCount * 5;
		List<SplitAction> splitActions = new List<SplitAction>(actionEstimate);
		
		//We want to count how many vertices are yielded from each triangle split. This will be used later to add the indices.
		short[] frontVertexCount = new short[inputTriangleCount];
		short[] backVertexCount = new short[inputTriangleCount];
		
		short totalFront = 0, totalBack = 0;
	
		for(int i = 0; i < sourceIndices.Length; i += 3)
		{
			int[] indices = { sourceIndices[i], sourceIndices[i+1], sourceIndices[i+2] };
			
			float[] sides = { pointClassifications[i], pointClassifications[i+1], pointClassifications[i+2] };
			
			short indexA = 2;
			
			short front = 0, back = 0;
			
			for(short indexB = 0; indexB < 3; indexB++)
			{
				float sideA = sides[indexA];
				float sideB = sides[indexB];
				
				if(sideB > 0f)
				{
					if(sideA < 0f)
					{
						//Find intersection between A, B. Add to BOTH
						splitActions.Add( new SplitAction(indices[indexA], indices[indexB], i) );
						front++;
						back++;
					}
					//Add B to FRONT.
					splitActions.Add( new SplitAction(true, false, indices[indexB]));
					front++;
				}
				else if (sideB < 0f)
				{
					if (sideA > 0f)
					{
						//Find intersection between A, B. Add to BOTH
						splitActions.Add( new SplitAction(indices[indexA], indices[indexB], i));
						front++;
						back++;
					}
					//Add B to BACK.
					splitActions.Add( new SplitAction(false, true, indices[indexB]));
					back++;
				}
				else
				{
					//Add B to BOTH.
					splitActions.Add( new SplitAction(false, true,  indices[indexB]));
					front++;
					back++;
				}
				
				indexA = indexB;
			}
			
			int j = i / 3; //This is the triangle counter.
			
			frontVertexCount[j] = front;
			backVertexCount[j] = back;
			
			totalFront += front;
			totalBack += back;
		}
		
		// We're going to iterate through the splits only several times, so let's
		//find the subset once now.
		// Since these are STRUCTs, this is going to COPY the array content. The
		//intersectionInverseRelation table made below helps us put it back into the
		//main array before we use it.
		SplitAction[] intersectionActions;
		int[] intersectionInverseRelation;
		{
			int intersectionCount = 0;
			
			foreach(SplitAction sa in splitActions)
				if((sa.flags & SplitAction.INTERSECT) == SplitAction.INTERSECT)
					intersectionCount++;
			
			intersectionActions = new SplitAction[intersectionCount];
			intersectionInverseRelation = new int[intersectionCount];
			
			int j = 0;
			for(int i = 0; i < splitActions.Count; i++)
			{
				SplitAction sa = splitActions[i];
				if((sa.flags & SplitAction.INTERSECT) == SplitAction.INTERSECT)
				{
					intersectionActions[j] = sa;
					intersectionInverseRelation[j] = i;
					j++;
				}
			}
		}
		
		// Next, we're going to find out which splitActions replicate the work of other split actions.
		//A given SA replicates another if and only if it _both_ calls for an intersection _and_ has
		//the same two parent indices (index0 and index1). This is because all intersections are called
		//with the same other parameters, so any case with an index0 and index1 matching will yield the
		//same results.
		// Only caveat is that two given splitActions might as the source indices in reverse order, so
		//we'll arbitrarily decide that "greater first" or something is the correct order. Flipping this
		//order has no consequence until after the intersection is found (at which point flipping the order
		//necessitates converting intersection i to 1-i to flip it as well.)
		// We can assume that every SA has at most 1 correlation. For a given SA, we'll search the list
		//UP TO its own index and, if we find one, we'll take the other's index and put it into the CLONE OF
		//slot.
		// So if we had a set like AFDBAK, than when the _latter_ A comes up for assessment, it'll find
		//the _first_ A (with an index of 0) and set the latter A's cloneOf figure to 0. This way we know
		//any latter As are a clone of the first A.
		
		for(int i = 0; i < intersectionActions.Length; i++)
		{
			SplitAction a = intersectionActions[i];
			
			//Ensure that the index0, index1 figures are all in the same order.
			//(We'll do this as we walk the list.)
			if(a.index0 > a.index1)
			{
				int j = a.index0;
				a.index0 = a.index1;
				a.index1 = j;
			}
			
			Vector3 aVector = sourceGeometry[a.index0] + sourceGeometry[a.index1];
			
			//Only latters clone formers, so we don't need to search up to and past the self.
			for(int j = 0; j < i; j++)
			{
				SplitAction b = intersectionActions[j];
				
				bool match = a.index0 == b.index0 && a.index1 == b.index1;
				
				//TEMPORARY HACK
				//
				// Infill requires that we match doubled vertices based on their physical
				//position and needs a purely-geometrical analysis of this. However as the
				//kit is currently architected, this data will also be used for the slice
				//geometry.
				// This means that UVs will be mangled as they're not taken into account.
				//This stopgap fix makes it so only matches doubles if infill is actually
				//activated. There may be some distorted where UVs are unwelded, but on
				//typical models this will be minor.
				//
				if(doInfill)
				{
					if(!match)
					{
						Vector3 bVector = sourceGeometry[b.index0] + sourceGeometry[b.index1];
						
						match = aVector == bVector;
					}
				}
				
				if(match)
				{
					a.cloneOf = j;
				}
			}
			
			intersectionActions[i] = a;
		}
		
		//Next, we want to perform all INTERSECTIONS. Any action which has an intersection needs to have that, like, done.
		
		for(int i = 0; i < intersectionActions.Length; i++)
		{
			SplitAction sa = intersectionActions[i];
			
			if(sa.cloneOf == SplitAction.nullIndex)
			{
				Vector3 pointA = sourceGeometry[ sa.index0 ];
				Vector3 pointB = sourceGeometry[ sa.index1 ];
				sa.intersectionResult = intersectCommon(ref pointB, ref pointA, ref plane);
				intersectionActions[i] = sa;
			}
		}
		
		// Let's create a table that relates an INTERSECTION index to a GEOMETRY index with an offset of 0 (for example
		//to refer to our newVertices or to the transformedVertices or whatever; internal use.)
		// We can also set up our realIndex figures in the same go.
		int newIndexStartsAt = meshCache.vertices.Count;
		int uniqueVertexCount = 0;
		int[] localIndexByIntersection = new int[intersectionActions.Length];
		{
			int currentLocalIndex = 0;
			for(int i = 0; i < intersectionActions.Length; i++)
			{
				SplitAction sa = intersectionActions[i];
				
				int j;
				
				if(sa.cloneOf == SplitAction.nullIndex)
				{
					j = currentLocalIndex++;
				}
				else
				{
					//This assumes that the widget that we are a clone of already has its localIndexByIntersection assigned.
					//We assume this because above – where we seek for clones – we only look behind for cloned elements.
					j = localIndexByIntersection[sa.cloneOf];
				}
				
				sa.realIndex = newIndexStartsAt + j;
				
				localIndexByIntersection[i] = j;
				
				intersectionActions[i] = sa;
			}
			uniqueVertexCount = currentLocalIndex;
		}
		
		//Let's figure out how much geometry we might have.
		//The infill geometry is a pair of clones of this geometry, but with different NORMALS and UVs. (Each set has different normals.)
		
		int newGeometryEstimate = uniqueVertexCount * (doInfill ? 3 : 1);
		
		//In this ACTION pass we'll act upon intersections by fetching both referred vertices and LERPing as appropriate.
		//The resultant indices will be written out over the index0 figures.
		
		Vector3[] newVertices = new Vector3[newGeometryEstimate];
		#if NORMALS
		Vector3[] newNormals = new Vector3[newGeometryEstimate];
		#endif
		Vector2[] newUVs = new Vector2[newGeometryEstimate];
			
		//LERP to create vertices
		{
			int currentNewIndex = 0;
			foreach(SplitAction sa in intersectionActions)
			{
				if(sa.cloneOf == SplitAction.nullIndex)
				{
					Vector3 v = sourceGeometry[sa.index0];
					Vector3 v2 = sourceGeometry[sa.index1];
					newVertices[currentNewIndex] = Vector3.Lerp(v2, v, sa.intersectionResult);
					currentNewIndex++;
				}
			}
		}
		
		//Normals:
		#if NORMALS
		{
			int currentNewIndex = 0;
			foreach(SplitAction sa in intersectionActions)
			{	
				if(sa.cloneOf == SplitAction.nullIndex)
				{
					Vector3 n = sourceNormals[sa.index0];
					Vector3 n2 = sourceNormals[sa.index1];
					newNormals[currentNewIndex] = Vector3.Lerp(n2, n, sa.intersectionResult);
					currentNewIndex++;
				}
			}
		}
		#endif
		
		//UVs:
		{
			int currentNewIndex = 0;
			foreach(SplitAction sa in intersectionActions)
			{
				if(sa.cloneOf == SplitAction.nullIndex)
				{
					Vector2 uv = sourceUVs[sa.index0];
					Vector2 uv2 = sourceUVs[sa.index1];
					newUVs[currentNewIndex] = Vector2.Lerp(uv2, uv, sa.intersectionResult);
					currentNewIndex++;
				}
			}
		}
		
		//All the polygon triangulation algorithms depend on having a 2D polygon. We also need the slice plane's
		//geometry in two-space to map the UVs.
		
		//NOTE that as we only need this data to analyze polygon geometry for triangulation, we can TRANSFORM (scale, translate, rotate)
		//these figures any way we like, as long as they retain the same relative geometry. So we're going to perform ops on this
		//data to create the UVs by scaling it around, and we'll feed the same data to the triangulator.
		
		//Our's exists in three-space, but is essentially flat... So we can transform it onto a flat coordinate system.
		//The first three figures of our plane four-vector describe the normal to the plane, so if we can create
		//a transformation matrix from that normal to the up normal, we can transform the vertices for observation.
		//We don't need to transform them back; we simply refer to the original vertex coordinates by their index,
		//which (as this is an ordered set) will match the indices of coorisponding transformed vertices.
		
		//This vector-vector transformation comes from Benjamin Zhu at SGI, pulled from a 1992
		//forum posting here: http://steve.hollasch.net/cgindex/math/rotvecs.html
		
		/*	"A somewhat "nasty" way to solve this problem:

			Let V1 = [ x1, y1, z1 ], V2 = [ x2, y2, z2 ]. Assume V1 and V2 are already normalized.
			
			    V3 = normalize(cross(V1, V2)). (the normalization here is mandatory.)
			    V4 = cross(V3, V1).
			             
			         [ V1 ]
			    M1 = [ V4 ]
			         [ V3 ]
			
			    cos = dot(V2, V1), sin = dot(V2, V4)
			            
			         [ cos   sin    0 ]
			    M2 = [ -sin  cos    0 ]
			         [ 0     0      1 ]
			         
			The sought transformation matrix is just M1^-1 * M2 * M1. This might well be a standard-text solution."
			
			-Ben Zhu, SGI, 1992
		 */
		
		Vector2[] transformedVertices = new Vector2[0];
		int infillFrontOffset = 0, infillBackOffset = 0;
		
		if(doInfill)
		{
			transformedVertices = new Vector2[newGeometryEstimate];
			
			Matrix4x4 flattenTransform;
			
			//Based on the algorithm described above, this will create a matrix permitting us
			//to multiply a given vertex yielding a vertex transformed to an XY plane (where Z is
			//undefined.)
			{
				Vector3 v1 = Vector3.forward;
				Vector3 v2 = new Vector3( plane.x, plane.y, plane.z ).normalized;
				Vector3 v3 = Vector3.Cross( v1, v2 ).normalized;
				Vector3 v4 = Vector3.Cross( v3, v1 );
				
				float cos = Vector3.Dot(v2, v1);
				float sin = Vector3.Dot(v2, v4);
				
				Matrix4x4 m1 = Matrix4x4.identity;
				m1.SetRow(0, (Vector4) v1);
				m1.SetRow(1, (Vector4) v4);
				m1.SetRow(2, (Vector4) v3);
				
				Matrix4x4 m1i = m1.inverse;
				
				Matrix4x4 m2 = Matrix4x4.identity;
				m2.SetRow(0, new Vector4(cos, sin, 0, 0) );
				m2.SetRow(1, new Vector4(-sin, cos, 0, 0) );
				
				flattenTransform = m1i * m2 * m1;
			}
			
			for(int i = 0; i < newVertices.Length; i++)
			{
				transformedVertices[i] = (Vector2) flattenTransform.MultiplyPoint3x4( newVertices[i] );
			}
			
			// We want to normalize the entire transformed vertices. To do this, we find the largest
			//floats in either (by abs). Then we scale. Of course, this normalizes us to figures
			//in the range of [-1f,1f] (not necessarily extending all the way on both sides), and
			//what we need are figures between 0f and 1f (not necessarily filling, but necessarily
			//not spilling.) So we'll shift it here.
			{
				float x = 0f, y = 0f;
				
				for(int i = 0; i < transformedVertices.Length; i++)
				{
					Vector2 v = transformedVertices[i];
					
					v.x = Mathf.Abs(v.x);
					v.y = Mathf.Abs(v.y);
					
					if(v.x > x) x = v.x;
					if(v.y > y) y = v.y;
				}
				
				//We would use 1f/x, 1f/y but we also want to scale everything to half (and perform an offset) as
				//described above.
				x = 0.5f / x;
				y = 0.5f / y;
				
				Rect r = infill.regionForInfill;
				
				for(int i = 0; i < transformedVertices.Length; i++)
				{
					Vector2 v = transformedVertices[i];
					v.x *= x;
					v.y *= y;
					v.x += 0.5f;
					v.y += 0.5f;
					v.x *= r.width;
					v.y *= r.height;
					v.x += r.x;
					v.y += r.y;
					transformedVertices[i] = v;
				}
			}
			
			//Now let's build the geometry for the two slice in-fills.
			//One is for the front side, and the other for the back side. Each has differing normals.
			
			infillFrontOffset = uniqueVertexCount;
			infillBackOffset = uniqueVertexCount * 2;
			
			//The geometry is identical...
			
			System.Array.Copy(newVertices, 0, newVertices, infillFrontOffset, uniqueVertexCount);
			System.Array.Copy(newVertices, 0, newVertices, infillBackOffset, uniqueVertexCount);
			
			System.Array.Copy(transformedVertices, 0, newUVs, infillFrontOffset, uniqueVertexCount);
			System.Array.Copy(transformedVertices, 0, newUVs, infillBackOffset, uniqueVertexCount);
			
			#if NORMALS
			
			Vector3 infillFrontNormal = ((Vector3) plane) * -1f;
			infillFrontNormal.Normalize();
			
			for(int i = infillFrontOffset; i < infillBackOffset; i++)
				newNormals[i] = infillFrontNormal;
			
			Vector3 infillBackNormal = (Vector3) plane;
			infillBackNormal.Normalize();
			
			for(int i = infillBackOffset; i < newNormals.Length; i++)
				newNormals[i] = infillBackNormal;
			
			#endif
		}
		
		//Get the exact indices into two tables. Note that these are indices for TRIANGLES and QUADS, which we'll triangulate in the next section.
		int[] newFrontIndex = new int[totalFront];
		int[] newBackIndex = new int[totalBack];
		
		//Note that here we refer to split actions again, so let's copy back the updated splitActions.
		for(int i = 0; i < intersectionActions.Length; i++)
		{
			int j = intersectionInverseRelation[i];
			splitActions[j] = intersectionActions[i];
		}
		
		int newFrontIndexCount = 0, newBackIndexCount = 0;
		foreach(SplitAction sa in splitActions)
		{
			if((sa.flags & SplitAction.TO_FRONT) == SplitAction.TO_FRONT)
			{
				newFrontIndex[newFrontIndexCount] = sa.realIndex;
				newFrontIndexCount++;
			}
			if((sa.flags & SplitAction.TO_BACK) == SplitAction.TO_BACK)
			{
				newBackIndex[newBackIndexCount] = sa.realIndex;
				newBackIndexCount++;
			}
		}
		
		//Now we need to triangulate sets of quads.
		//We recorded earlier whether we're looking at triangles or quads – in order. So we have a pattern like TTQTTQQTTTQ, and
		//we can expect these vertices to match up perfectly to what the above section of code dumped out.
		
		int startIndex = 0;
		
		int[] _indices3 = new int[3];
		int[] _indices4 = new int[6];
		
		foreach(short s in frontVertexCount)
		{
			if(s == 3)
			{
				_indices3[0] = newFrontIndex[startIndex];
				_indices3[1] = newFrontIndex[startIndex + 1];
				_indices3[2] = newFrontIndex[startIndex + 2];
				frontIndices.AddArray(_indices3);
			}
			else if(s == 4)
			{
				_indices4[0] = newFrontIndex[startIndex];
				_indices4[1] = newFrontIndex[startIndex + 1];
				_indices4[2] = newFrontIndex[startIndex + 3];
				_indices4[3] = newFrontIndex[startIndex + 1];
				_indices4[4] = newFrontIndex[startIndex + 2];
				_indices4[5] = newFrontIndex[startIndex + 3];
				frontIndices.AddArray(_indices4);
			}
			startIndex += s;
		}
		
		startIndex = 0;
		
		foreach(short s in backVertexCount)
		{
			if(s == 3)
			{
				_indices3[0] = newBackIndex[startIndex];
				_indices3[1] = newBackIndex[startIndex + 1];
				_indices3[2] = newBackIndex[startIndex + 2];
				backIndices.AddArray(_indices3);
			}
			else if(s == 4)
			{
				_indices4[0] = newBackIndex[startIndex];
				_indices4[1] = newBackIndex[startIndex + 1];
				_indices4[2] = newBackIndex[startIndex + 3];
				_indices4[3] = newBackIndex[startIndex + 1];
				_indices4[4] = newBackIndex[startIndex + 2];
				_indices4[5] = newBackIndex[startIndex + 3];
				backIndices.AddArray(_indices4);
			}
			startIndex += s;
		}
		
		//Let's add this shiznit in!
		
		meshCache.vertices.AddArray(newVertices);
		#if NORMALS
		meshCache.normals.AddArray(newNormals);
		#endif
		meshCache.UVs.AddArray(newUVs);

		//Now we need to fill in the slice hole.
		
		//We need to find the POLYGON[s] representing the slice hole[s]. There may be more than one. 
		//Then we need to TRIANGULATE these polygons and write them out.
		
		//Above we've built the data necessary to pull this off. We have:
		
		// - Geometry for the polygon around the edges in Vertex3 / Normal / UV format, already added
		//to the geometry setup.
		// - Geometry for the polygon in Vertex2 format in matching order, aligned to the slice plane.
		// - A collection of all data points and 1:1 hashes representing their physical location.
		
		//In this mess of data here may be 0 or non-zero CLOSED POLYGONS. We need to walk the list and
		//identify each CLOSED POLYGON (there may be none, or multiples). Then, each of these must be
		//triangulated separately.
		
		//Vertices connected to each other in a closed polygon can be found to associate with each other
		//in two ways. Envision a triangle strip that forms a circular ribbon – and that we slice through
		//the middle of this ribbon. Slice vertices come in two kinds of pairs; there are pairs that COME FROM
		//the SAME triangle, and pairs that come from ADJACENT TRIANGLES. The whole chain is formed from
		//alternating pair-types.
		
		//So for example vertex A comes from the same triangle as vertex B, which in turn matches the position
		//of the NEXT triangle's vertex A.
		
		//The data is prepared for us to be able to identify both kinds of associations. First,
		//association by parent triangle is encoded in the ORDERING. Every PAIR from index 0 shares a parent
		//triangle; so indices 0-1, 2-3, 4-5 and so on are each a pair from a common parent triangle.
		
		//Meanwhile, vertices generated from the common edge of two different triangles will have the SAME
		//POSITION in three-space.
		
		//We don't have to compare Vector3s, however; this has already been done. Uniques were eliminated above.
		//What we have is a table; localIndexByIntersection. This list describes ALL SLICE VERTICES in terms
		//of which VERTEX (in the array – identified by index) represents that slice vertex. So if we see that
		//localIndexByIntersection[0] == localIndexByIntersection[4], than we know that slice vertices 0 and 4
		//share the same position in three space.
		
		//With that in mind, we're going to go through the list in circles building chains out of these
		//connections.
		
		if(doInfill)
		{
			List<int> currentWorkingPoly = new List<int>();
			List<int> currentTargetPoly = new List<int>();
			List<List<int>> allPolys = new List<List<int>>();
			List<int> claimed = new List<int>();
			
			int lastAdded = -1;
			
			//ASSUMPTION: Every element will be claimed into some kind of chain by the end whether correlated or not.
			do
			{
				for(int i = 0; i < localIndexByIntersection.Length; i++)
				{
					bool go = false, fail = false, startNewChain = false;
					
					//If we didn't just add one, we're looking to start a chain. That means we have to find one that
					//isn't already claimed.
					if(lastAdded < 0)
					{
						go = claimed.Contains(i) == false;
					}
					else if(lastAdded == i)
					{
						//We've gone through twice without finding a match. This means there isn't one, or something.
						
						fail = true;
					}
					else
					{
						//Otherwise, we're trying to find the next-in-chain.
						//A valid next-in-chain is connected by geometry which, as discussed, means it's connected
						//by having matching parent indices (index0, index1).
						
						bool match = localIndexByIntersection[i] == localIndexByIntersection[lastAdded];
						
						//But there's a special case about the match; it's possible that we've closed the loop!
						//How do we know we've closed the loop? There are multiple ways but the simplest is that
						//the chain already contains the element in question.
						
						bool loopComplete = match && currentWorkingPoly.Contains(i);
						
						if(loopComplete)
						{
							allPolys.Add(currentTargetPoly);
							startNewChain = true;
						}
						else
						{
							go = match;
						}
					}
					
					if(go)
					{
						int partnerByParent = i % 2 == 1 ? i - 1 : i + 1;
						
						int[] pair = { i, partnerByParent };
						
						currentWorkingPoly.AddRange(pair);
						claimed.AddRange(pair);
						
						currentTargetPoly.Add(partnerByParent);
						
						lastAdded = partnerByParent;
						
						//Skip ahead and resume the search _from_ here, so that we don't step into it
						//again from within this loop walk.
						i = partnerByParent;
					}
					else if(fail)
					{
						//We want to start a fresh poly without adding this to the valid polys.
						startNewChain = true;
						
						//Debug.Log("[fail]");
					}
					
					if(startNewChain)
					{
						currentWorkingPoly.Clear();
						currentTargetPoly = new List<int>();
						lastAdded = -1;
					}
				}
			}
			while(currentWorkingPoly.Count > 0);
			
			//Now we go through each poly and triangulate it.
			
			foreach(List<int> _poly in allPolys)
			{
				Vector2[] poly = new Vector2[_poly.Count];
				
				for(int i = 0; i < poly.Length; i++)
				{
					int j = localIndexByIntersection[ _poly[i] ];
					poly[i] = transformedVertices[j];
				}
				
				int[] result;
				
				if(triangulate(poly, out result))
				{
					int[] front = new int[result.Length];
					int[] back = new int[result.Length];
					
					for(int i = 0; i < result.Length; i++)
					{
						int p = _poly[ result[i] ];
						int local = localIndexByIntersection[ p ];
						front[i] = local + infillFrontOffset + newIndexStartsAt;
						back[i] = local + infillBackOffset + newIndexStartsAt;
					}
					
					for(int i = 0; i < result.Length; i += 3)
					{
						int j = front[i];
						front[i] = front[i + 2];
						front[i + 2] = j;
					}
					
					frontIndices.AddArray(front);
					backIndices.AddArray(back);
				}
				//else
				//{
					//There is some sort of edge case where the code feeding the triangulator will spit out repeating vertices.
					//It could be anywhere above – or it could even be that the slicer itself is spitting junk data into its
					//child objects which confuses subsequent processes. It is worth noting that it mainly seems to occur on very
					//small objects.
				//}
			}
		}
	}
	
	//These here are the only public interfaces.
	//You can see that the core interface is splitByPlane and the others merely wrap it; the math uses Vector4 plane descriptions
	//throughout, so the byLine and byTriangle are conveniences.
	
	public GameObject[] splitByLine(GameObject target, Camera camera, Vector3 _start, Vector3 _end)
	{
		return splitByLine(target, camera, _start, _end, true);
	}
	
	public GameObject[] splitByLine(GameObject target, Camera camera, Vector3 _start, Vector3 _end, bool destroyOriginal)
	{
		Vector4 v = planeForTransform(camera, target, _start, _end);
		return splitByPlane(target, v, destroyOriginal);
	}
	
	public GameObject[] splitByTriangle(GameObject target, Vector3[] triangleInWorldSpace, bool destroyOriginal)
	{
		Vector4 v = TurboSlice.planeFromTriangle(target, triangleInWorldSpace);
		return splitByPlane(target, v, destroyOriginal);
	}
	
	public GameObject[] shatter(GameObject go, int steps)
	{
		List<GameObject> l = new List<GameObject>(1);
		
		l.Add(go);
		
		List<GameObject> l2 = l;
		
		for(int i = 0; i < steps; i++)
		{
			l2 = new List<GameObject>(l.Count * 2);
			
			Vector4 shatterPlane = (Vector4) Random.insideUnitSphere;
			
			foreach(GameObject go2 in l)
			{
				l2.AddRange( splitByPlane(go2, shatterPlane, true) );
			}
			
			l = l2;
		}
		
		return l2.ToArray();
	}
	
	public GameObject[] splitByPlane(GameObject go, Vector4 plane, bool destroyOriginal)
	{
		if(go.GetComponentInChildren<SkinnedMeshRenderer>() != null)
		{
			return splitByPlaneRD(go, plane, destroyOriginal);
		}
	
		MeshCache c;
		
		Sliceable sliceable = null;
		InfillConfiguration[] ourInfills = infills;
		bool yieldClones = false;
		
		{
			Mesh m = null;
			
			MeshFilter[] filters = go.GetComponentsInChildren<MeshFilter>();
			
			if(filters.Length < 1)
			{
				Debug.LogWarning("TurboSlice cannot find mesh filter in object '" + go.name + "' in scene '" + Application.loadedLevelName + "'! Only objects featuring a single mesh filter can be sliced.");
				
				GameObject[] result = { go };
				
				return result;
			}
			else
			{
//				if(filters.Length > 1)
//				{
//					Debug.LogWarning("TurboSlice found multiple mesh filters in object '" + go.name + "' in scene '" + Application.loadedLevelName + "'! Behavior is undefined.");				
//				}
				
				m = filters[0].sharedMesh;
			}
			
			if(meshCaches != null && meshCaches.ContainsKey(m))
			{
				c = meshCaches[m];
				
				//The mesh cache will be directly modified under the assumption that this will be discarded shortly
				//and thus picked up by the GC. It will grow in size; it will not shrink. Thus we do not want to
				//operate on the original, semi-persistent mesh caches that were preloaded on boot. Instead, we want
				//to make a clone.
				
				if(c.wasPreloaded)
				{
					c = c.clone();
				}
			}
			else
				c = cacheFromGameObject(go, true);
			
			Sliceable[] sliceables = go.GetComponentsInChildren<Sliceable>();
			
			if(sliceables.Length > 0)
			{
				if(sliceables.Length > 1)
				{
					Debug.LogWarning("TurboSlice found multiple slice configurations in object '" + go.name + "' in scene '" + Application.loadedLevelName + "'! Behavior is undefined.");				
				}
				
				sliceable = sliceables[0];
				
				if(sliceable.infillers.Length > 0)
					ourInfills = sliceable.infillers;
				
				yieldClones = sliceable.yieldClones;
			}
			else
			{
				sliceable = go.AddComponent<Sliceable>();
				sliceable.yieldClones = yieldClones;
				sliceable.currentlySliceable = true;
				sliceable.refreshColliders = true;
			}
		}
		
		if(sliceable != null && !sliceable.currentlySliceable)
		{
			GameObject[] result = { go };
			
			return result;
		}
		
		int submeshCount = c.indices.Length;
		
		//We're going to create two new tentative meshes which contain ALL original vertices in order,
		//plus room for new vertices. Not all of these copied vertices will be addressed, but copying them
		//over eliminates the need to remove doubles and do an On^2 search.
		
		TurboList<int>[] _frontIndices = new TurboList<int>[ submeshCount ];
		TurboList<int>[] _backIndices = new TurboList<int>[ submeshCount ];
		
		PlaneTriResult[] sidePlanes = new PlaneTriResult[c.vertices.Count];
		{
			Vector3[] vertices = c.vertices.array;
			
			for(int i = 0; i < sidePlanes.Length; i++)
			{
				sidePlanes[i] = getSidePlane(ref vertices[i], ref plane);
			}
		}
		
		for(int j = 0; j < submeshCount; j++)
		{	
			int initialCapacityIndices = Mathf.RoundToInt((float) c.indices[j].Length * factorOfSafetyIndices);
		
			_frontIndices[j] = new TurboList<int>(initialCapacityIndices);
			_backIndices[j] = new TurboList<int>(initialCapacityIndices);
			
			int[] _indices = c.indices[j];
			
			TurboList<int> frontIndices = _frontIndices[j];
			TurboList<int> backIndices = _backIndices[j];
			TurboList<int> splitPending = new TurboList<int>(initialCapacityIndices);
		
			int[] indices = new int[3];
			
			for(int i = 0; i < _indices.Length; )
			{	
				indices[0] = _indices[i++];
				indices[1] = _indices[i++];
				indices[2] = _indices[i++];
				
				// compute the side of the plane each vertex is on
				PlaneTriResult r1 = sidePlanes[indices[0]];
				PlaneTriResult r2 = sidePlanes[indices[1]];
				PlaneTriResult r3 = sidePlanes[indices[2]];
				
				if ( r1 == r2 && r1 == r3 ) // if all three vertices are on the same side of the plane.
				{
					if ( r1 == PlaneTriResult.PTR_FRONT ) // if all three are in front of the plane, then copy to the 'front' output triangle.
					{
						frontIndices.AddArray(indices);
					}
					else
					{
						backIndices.AddArray(indices);
					}
				}
				else
				{
					splitPending.AddArray(indices);
				}
			}
			
			InfillConfiguration ifc = null;
			
			if(j < c.mats.Length)
			{
				Material mat = c.mats[j];
				
				foreach(InfillConfiguration _ifc in ourInfills)
				{
					if(_ifc.material == mat)
					{
						ifc = _ifc;
					}
				}
			}

			splitTriangles(plane, splitPending.ToArray(), c, ifc, frontIndices, backIndices);
		}
		
		GameObject[] results;
		
		bool onlyHaveOne = true;
		
		for(int i = 0; i < c.indices.Length; i++)
		{
			onlyHaveOne &= _frontIndices[i].Count == 0 || _backIndices[i].Count == 0;
		}

		if(onlyHaveOne)
		{	
			//Do nothing
			results = new GameObject[1];
			results[0] = go;
		}
		else
		{
			MeshCache frontCache = new MeshCache();
			//frontCache.creationTime = c.creationTime; //The creation time follows the lists.
			frontCache.vertices = c.vertices;
			#if NORMALS
			frontCache.normals = c.normals;
			#endif
			frontCache.UVs = c.UVs;
			frontCache.mats = c.mats;
			
			MeshCache backCache = new MeshCache();
			//backCache.creationTime = c.creationTime;
			backCache.vertices = c.vertices;
			#if NORMALS
			backCache.normals = c.normals;
			#endif
			backCache.UVs = c.UVs;
			backCache.mats = c.mats;
			
			frontCache.indices = new int[submeshCount][];
			backCache.indices = new int[submeshCount][];
			for(int i = 0; i < submeshCount; i++)
			{
				frontCache.indices[i] = _frontIndices[i].ToArray();
				backCache.indices[i] = _backIndices[i].ToArray();
			}
			
			Vector3[] geoSubsetOne, geoSubsetTwo;
			#if NORMALS
			Vector3[] normalsSubsetOne, normalsSubsetTwo;
			#endif
			Vector2[] uvSubsetOne, uvSubsetTwo;
			int[][] indexSubsetOne, indexSubsetTwo;
			
			indexSubsetOne = new int[submeshCount][];
			indexSubsetTwo = new int[submeshCount][];
			
			//Perfect subset will inflate the array list size if needed to the exact figure. So if we estimate 0,
			//and there is 1 submesh, than we will have 1 allocation, and this is optimal. Estimation can only help
			//if we have THREE or more submeshes, which is a silly scenario for anyone concerned about performance.
			int estimateOne = 0, estimateTwo = 0;
			
			TurboList<Vector3> _geoSubsetOne = new TurboList<Vector3>(estimateOne);
			TurboList<Vector3> _geoSubsetTwo = new TurboList<Vector3>(estimateTwo);
			TurboList<Vector3> _normalSubsetOne = new TurboList<Vector3>(estimateOne);
			TurboList<Vector3> _normalSubsetTwo = new TurboList<Vector3>(estimateTwo);
			TurboList<Vector2> _uvSubsetOne = new TurboList<Vector2>(estimateOne);
			TurboList<Vector2> _uvSubsetTwo = new TurboList<Vector2>(estimateTwo);
			
			int transferTableMaximumKey = c.vertices.Count;
			
			int[] transferTableOne = new int[transferTableMaximumKey];
			int[] transferTableTwo = new int[transferTableMaximumKey];
			
			for(int i = 0; i < transferTableOne.Length; i++) transferTableOne[i] = -1;
			for(int i = 0; i < transferTableTwo.Length; i++) transferTableTwo[i] = -1;
			
#if NORMALS
			for(int i = 0; i < submeshCount; i++)
				perfectSubset(_frontIndices[i], c.vertices, c.normals, c.UVs, out indexSubsetOne[i], _geoSubsetOne, _normalSubsetOne, _uvSubsetOne, ref transferTableOne );
			
			for(int i = 0; i < submeshCount; i++)
				perfectSubset(_backIndices[i], c.vertices, c.normals, c.UVs, out indexSubsetTwo[i], _geoSubsetTwo, _normalSubsetTwo, _uvSubsetTwo, ref transferTableTwo );
#else
			for(int i = 0; i < submeshCount; i++)
				perfectSubset(_frontIndices[i], c.vertices, c.UVs, out indexSubsetOne[i], _geoSubsetOne, _uvSubsetOne, ref transferTableOne );
			
			for(int i = 0; i < submeshCount; i++)
				perfectSubset(_backIndices[i], c.vertices,  c.UVs, out indexSubsetTwo[i], _geoSubsetTwo, _uvSubsetTwo, ref transferTableTwo );
#endif
			
			geoSubsetOne = _geoSubsetOne.ToArray();
			geoSubsetTwo = _geoSubsetTwo.ToArray();
#if NORMALS
			normalsSubsetOne = _normalSubsetOne.ToArray();
			normalsSubsetTwo = _normalSubsetTwo.ToArray();
#endif
			uvSubsetOne = _uvSubsetOne.ToArray();
			uvSubsetTwo = _uvSubsetTwo.ToArray();
			
			//Note that we do not explicitly call recalculate bounds because (as per the manual) this is implicit in an
			//assignment to vertices whenever the vertex count changes from zero to non-zero.
			
			
			Mesh frontMesh = new Mesh();
			Mesh backMesh = new Mesh();
			
			GameObject frontObject, backObject;
			
			createResultObjects(go, sliceable, false, plane, out frontObject, out backObject);
			
			frontObject.GetComponentInChildren<MeshFilter>().mesh = frontMesh;
			backObject.GetComponentInChildren<MeshFilter>().mesh = backMesh;
			
			frontMesh.vertices = geoSubsetOne;
			backMesh.vertices = geoSubsetTwo;
			
#if NORMALS
			frontMesh.normals = normalsSubsetOne;
			backMesh.normals = normalsSubsetTwo;
#endif
			frontMesh.uv = uvSubsetOne;
			backMesh.uv = uvSubsetTwo;
			
			frontMesh.subMeshCount = submeshCount;
			backMesh.subMeshCount = submeshCount;
			
			for(int i = 0 ; i < submeshCount; i++)
			{
				frontMesh.SetTriangles(indexSubsetOne[i], i);
				backMesh.SetTriangles(indexSubsetTwo[i], i);
			}

			if(meshCaches != null)
			{
				if(!yieldClones || go.GetComponent<DeletionCallback>() == null)
				{
					frontObject.AddComponent<DeletionCallback>();
					backObject.AddComponent<DeletionCallback>();
				}
					
				DeletionCallback frontCallback = frontObject.GetComponent<DeletionCallback>();
				DeletionCallback backCallback = backObject.GetComponent<DeletionCallback>();
				
				frontCallback.deletionListener = new DeletionOccurred(this.releaseCacheByMesh);
				backCallback.deletionListener = new DeletionOccurred(this.releaseCacheByMesh);
				
				frontCallback.mesh = frontMesh;
				backCallback.mesh = backMesh;
				
				meshCaches[frontMesh] = frontCache;
				meshCaches[backMesh] = backCache;
			}
			else
			{
				DeletionCallback frontCallback = frontObject.GetComponent<DeletionCallback>();
				DeletionCallback backCallback = backObject.GetComponent<DeletionCallback>();
				
				if(frontCallback != null)
					GameObject.DestroyImmediate(frontCallback);
				
				if(backCallback != null)
					GameObject.DestroyImmediate(backCallback);
			}
			
			if(destroyOriginal)
				GameObject.Destroy(go);
			
			results = new GameObject[2];
			results[0] = frontObject;
			results[1] = backObject;
			
			if(sliceable != null && sliceable.refreshColliders)
			{
				foreach(GameObject r in results)
				{
					Collider collider = r.collider;
					
					if(collider != null)
					{
						if(collider is BoxCollider)
						{
							GameObject.DestroyImmediate(collider);
							r.AddComponent<BoxCollider>();
						}
						else if(collider is SphereCollider)
						{
							GameObject.DestroyImmediate(collider);
							r.AddComponent<SphereCollider>();
						}
						else if(collider is MeshCollider)
						{
							MeshCollider mc = (MeshCollider) collider;
							
							bool isFront = r == frontObject;
							
							Mesh mesh = isFront ? frontMesh : backMesh;
							
							mc.sharedMesh = mesh;
						}
					}
				}
			}
			
			if(sliceable != null)
				sliceable.handleSlice(results);
		}
		
		return results;
	}
	
	private const float meshCacheSoftTimeout = 5f, meshCacheHardTimeout = 15f;
	private bool meshCacheWarningDelivered = false;
	
	// Update is called once per frame
	void Update ()
	{
		if(meshCaches == null)
			return;
		
		float t = Time.time;
		
		foreach(MeshCache mc in meshCaches.Values)
		{
			float age = t - mc.creationTime;
			bool softTimeout = age > meshCacheSoftTimeout;
			bool hardTimeout = age > meshCacheHardTimeout;
			
			if(!meshCacheWarningDelivered && softTimeout)
			{
				Debug.LogWarning("A mesh cache lingered over " + meshCacheSoftTimeout + " seconds. Please review the TurboSlice guide's performance section.");
				meshCacheWarningDelivered = true;
			}
			else if(hardTimeout)
			{
				Debug.LogWarning("A mesh cache lingered over " + meshCacheHardTimeout + " seconds. Disabling caches!");
				
				meshCaching = false;
				meshCaches = null;
				return;
			}
		}
	}
}
